/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QTimer>

#include "machine_offline.h"

static const int						TIMER_INTERVAL(40);

TMachineOffline::TMachineOffline(void)
{
}

TMachineOffline::~TMachineOffline(void)
{
}

void TMachineOffline::Process_Rx(
	const QByteArray					&new_data)
{
	Q_UNUSED(new_data);
}

QByteArray TMachineOffline::KeyPress_Done(void) const
{
	return QByteArray();
}

QByteArray TMachineOffline::KeyPress_DelPnt(void) const
{
	return QByteArray();
}

void TMachineOffline::Set_EStop(
	const bool 							state)
{
	d_estop_status = state;
	
	if(d_estop_status)
	{
		d_command_executor.Abort();
		
		d_jog_timer->stop();
		d_control_timer->stop();
		
		this->Send("ERR:  E-Stop Push-Button Pressed\r\n");
	}
}

bool TMachineOffline::Move_To(
	const TVector3						&pos)
{
	TCommandExecutor::TCommand			command;
	TVector3							target(pos);
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}

	target -= d_tool_offset;

	if(target.x < d_min_machine.x) target.x = d_min_machine.x;
	if(target.y < d_min_machine.y) target.y = d_min_machine.y;
	if(target.z < d_min_machine.z) target.z = d_min_machine.z;
	if(target.x > d_max_machine.x) target.x = d_max_machine.x;
	if(target.y > d_max_machine.y) target.y = d_max_machine.y;
	if(target.z > d_max_machine.z) target.z = d_max_machine.z;
	
	target += d_tool_offset;
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.target = pos;
	
	command.speed = d_move_speed;
	
	command.completion_text = QString();
	
	d_command_executor.Add_Command(command);
	
	if(!d_jog_timer->isActive())
	{
		d_jog_timer->start(TIMER_INTERVAL);
	}
	
	return true;
}

bool TMachineOffline::Manual_Touch(
	const TVector3 						&pos,
	const TVector3 						&vec)
{
	TCommandExecutor::TCommand			command;
	QString								text;
	TVector3							touch_pos(pos);
	TVector3							touch_vec(vec);
	TVector3							clear_pos;
	bool								x_error,y_error,z_error;
	static const double					ZERO_EPSILON(0.001);
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}
	
	// scale errors
	touch_pos.x += (pos.x * d_scale_factor_x);
	touch_pos.y += (pos.y * d_scale_factor_y);
	touch_pos.z += (pos.z * d_scale_factor_z);

	// squareness errors
	touch_pos.x += touch_pos.y * d_squareness_xy;
	touch_pos.y += touch_pos.z * d_squareness_yz;
	touch_pos.x += touch_pos.z * d_squareness_zx;

	if(touch_vec.Length() > ZERO_EPSILON)
	{
		touch_vec.Normal();
	}
	else
	{
		d_last_error = QStringLiteral("ERR:  Touch vector not defined");
		return false;
	}
	
	text = QString("Touch %1, %2, %3, %4, %5, %6")
				.arg(touch_pos.x,0,'f',3)
				.arg(touch_pos.y,0,'f',3)
				.arg(touch_pos.z,0,'f',3)
				.arg(touch_vec.i,0,'f',5)
				.arg(touch_vec.j,0,'f',5)
				.arg(touch_vec.k,0,'f',5);
	
	emit Log_Text(text);
	
	clear_pos = touch_pos;
	
	// touch IJK is approach but machine using surface normal
	touch_vec *= -1.0;
	
	text = QString("POINT %1, %2, %3, %4, %5, %6\r\n")
				.arg(touch_pos.x,0,'f',5)
				.arg(touch_pos.y,0,'f',5)
				.arg(touch_pos.z,0,'f',5)
				.arg(touch_vec.i,0,'f',5)
				.arg(touch_vec.j,0,'f',5)
				.arg(touch_vec.k,0,'f',5);
	
	if(this->Is_Position_Inside_Volume(clear_pos,&x_error,&y_error,&z_error))
	{
		command.command_type = TCommandExecutor::COMMAND_MOVE;
		command.target = clear_pos;
		command.speed = d_jog_highspeed;
		command.completion_text = text;
		
		d_command_executor.Add_Command(command);
		
		if(!d_control_timer->isActive())
		{
			d_control_timer->start(TIMER_INTERVAL);
		}
	}
	else
	{
		text = QString();
		
		if(x_error)
		{
			text += QStringLiteral("X");
		}
		
		if(y_error)
		{
			text += QStringLiteral("Y");
		}
		
		if(z_error)
		{
			text += QStringLiteral("Z");
		}
		
		d_last_error = QString("ERR:  Axis %1 outside of machine volume.").arg(text);
		
		
		return false;
	}
	
	return true;
}

bool TMachineOffline::Jog_Move(
	const TVector3						&direction)
{
	TCommandExecutor::TCommand			command;
	TVector3							target(d_command_executor.Current_Position());
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}
	
	if(d_control_timer->isActive())
	{
		d_last_error = QStringLiteral("ERR:  Machine currently active");
		return false;
	}
	
	target += (d_jog_distance * direction);

	target -= d_tool_offset;
	
	if(target.x < d_min_machine.x) target.x = d_min_machine.x;
	if(target.y < d_min_machine.y) target.y = d_min_machine.y;
	if(target.z < d_min_machine.z) target.z = d_min_machine.z;
	if(target.x > d_max_machine.x) target.x = d_max_machine.x;
	if(target.y > d_max_machine.y) target.y = d_max_machine.y;
	if(target.z > d_max_machine.z) target.z = d_max_machine.z;
	
	target += d_tool_offset;
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.target = target;
	
	command.speed = d_move_speed;
	
	command.completion_text = QString();
	
	d_command_executor.Add_Command(command);
	
	if(!d_jog_timer->isActive())
	{
		d_jog_timer->start(TIMER_INTERVAL);
	}
	
	return true;
}
