/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QOpenGLShaderProgram>
#include <QPainter>
#include <QRect>
#include <QColor>
#include <assert.h>

#include "../../core/openglshaders.h"

#include "opengldisplay.h"

static const double						ZERO_EPSILON = 0.0000001;

TOpenGLDisplay::TOpenGLDisplay(
    const QWidget						*parent,
    Qt::WindowFlags 					flags)
:TOpenGLWidget(const_cast<QWidget*>(parent),flags)
{
	d_touch_buffer_size = 0;

	d_machine_z_vertex_id = -1;
	d_machine_xz_vertex_id = -1;
	d_machine_x_vertex_id = -1;
	d_machine_xw_vertex_id = -1;
	d_machine_table_vertex_id = -1;

	d_machine_z_normal_id = -1;
	d_machine_xz_normal_id = -1;
	d_machine_x_normal_id = -1;
	d_machine_xw_normal_id = -1;
	d_machine_table_normal_id = -1;

	d_tool_stem_vertex_id = -1;
	d_probe_head_vertex_id = -1;
	d_tool_ruby_vertex_id = -1;
	
	d_tool_stem_normal_id = -1;
	d_probe_head_normal_id = -1;
	d_tool_ruby_normal_id = -1;

	d_touch_points_id = -1;
}

TOpenGLDisplay::~TOpenGLDisplay(void)
{
	d_opengl_vertex_buffer.Clear();
}

void TOpenGLDisplay::Define_Machine_Volume(
   const TVector3						&pnt_min,
   const TVector3						&pnt_max)
{
	TVector3							pt,vec;
	
	d_axis_draw_pos = pnt_min;
	
	d_min_xyz = pnt_min;
	d_max_xyz = pnt_max;
	
	pt = d_max_xyz - d_min_xyz;
	vec = pt;
	
	// find shortest vector
	d_axis_draw_len = pt.x;
	
	if(pt.y < d_axis_draw_len)
	{
		d_axis_draw_len = pt.y;
	}
	
	if(pt.z < d_axis_draw_len)
	{
		d_axis_draw_len = pt.z;
	}
	
	if(vec.Length() > ZERO_EPSILON)
	{
		vec.Normal();
		
		d_axis_draw_len /= 5;
		
		d_axis_draw_pos = d_min_xyz - vec * d_axis_draw_len;
		d_axis_draw_pos = d_min_xyz - vec * d_axis_draw_len;
	}
}

void TOpenGLDisplay::Set_Machine(const TMachine &machine)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>::const_iterator iter;
	double								*vertex;
	double								*normal;
	unsigned int						size;
	unsigned int						ucntr;
	
	size = machine.machine_z.size() * 3;
	
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_z_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_z_normal_id);
	
	if(size)
	{
		d_machine_z_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_machine_z_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_z_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_machine_z_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = machine.machine_z.begin();iter != machine.machine_z.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}
	
	size = machine.machine_xz.size() * 3;
	
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_xz_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_xz_normal_id);
	
	if(size)
	{
		d_machine_xz_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_machine_xz_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_xz_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_machine_xz_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = machine.machine_xz.begin();iter != machine.machine_xz.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}
	
	size = machine.machine_x.size() * 3;
	
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_x_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_x_normal_id);
	
	if(size)
	{
		d_machine_x_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_machine_x_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_x_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_machine_x_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = machine.machine_x.begin();iter != machine.machine_x.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}
	
	size = machine.machine_xw.size() * 3;
	
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_xw_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_xw_normal_id);
	
	if(size)
	{
		d_machine_xw_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_machine_xw_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_xw_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_machine_xw_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = machine.machine_xw.begin();iter != machine.machine_xw.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}
	
	size = machine.machine_table.size() * 3;
	
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_table_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_machine_table_normal_id);
	
	if(size)
	{
		d_machine_table_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_machine_table_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_table_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_machine_table_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = machine.machine_table.begin();iter != machine.machine_table.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}

}

void TOpenGLDisplay::Set_Tool(
	const TSensor 						&probe)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>::const_iterator iter;
	double								*vertex;
	double								*normal;
	unsigned int						size;
	unsigned int						ucntr;
	
	d_opengl_vertex_buffer.Remove_Buffer(d_tool_stem_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_probe_head_vertex_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_tool_ruby_vertex_id);
	
	d_opengl_vertex_buffer.Remove_Buffer(d_tool_stem_normal_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_probe_head_normal_id);
	d_opengl_vertex_buffer.Remove_Buffer(d_tool_ruby_normal_id);

	// tool_stem
	size = probe.tool_stem.size() * 3;

	if(size)
	{
		d_tool_stem_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_tool_stem_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_tool_stem_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_tool_stem_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = probe.tool_stem.begin();iter != probe.tool_stem.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}

	// probe_head
	size = probe.probe_head.size() * 3;
	
	if(size)
	{
		d_probe_head_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_probe_head_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_probe_head_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_probe_head_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = probe.probe_head.begin();iter != probe.probe_head.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}
	
	// ruby
	size = probe.tool_ruby.size() * 3;
	
	if(size)
	{
		d_tool_ruby_vertex_id = d_opengl_vertex_buffer.Create_Buffer(size);
		d_tool_ruby_normal_id = d_opengl_vertex_buffer.Create_Buffer(size);
		
		vertex = d_opengl_vertex_buffer.Buffer(d_tool_ruby_vertex_id);
		normal = d_opengl_vertex_buffer.Buffer(d_tool_ruby_normal_id);
		
		if(vertex && normal)
		{
			ucntr = 0;
			for(iter = probe.tool_ruby.begin();iter != probe.tool_ruby.end();++iter)
			{
				vertex[ucntr] = (*iter).vertex.x;
				vertex[ucntr+1] = (*iter).vertex.y;
				vertex[ucntr+2] = (*iter).vertex.z;
				
				normal[ucntr] = (*iter).normal.i;
				normal[ucntr+1] = (*iter).normal.j;
				normal[ucntr+2] = (*iter).normal.k;
				
				ucntr += 3;
			}
		}
	}
}

void TOpenGLDisplay::Path_To(
	const TVector3						&pos)
{
	if(d_touch_buffer.size())
	{
		if(d_touch_buffer[d_touch_buffer.size()-1].path.size() > 500)	// must cap the number of recorded moves on last touch point to something
		{
			d_touch_buffer.clear();
			
			d_opengl_vertex_buffer.Remove_Buffer(d_touch_points_id);
			d_touch_points_id = -1;
		}
		else
		{
			d_touch_buffer[d_touch_buffer.size()-1].path.push_back(pos);
		}
	}
}

void TOpenGLDisplay::Add_Touch(
	const TVector3						&pos)
{
	TOpenGLDisplay::TTouchPoint			touch_point;
	
	touch_point.pnt = pos;
	touch_point.path.push_back(pos);	// touch is also the first point of the path
	
	d_touch_buffer.push_back(touch_point);

	this->Update_Path_Buffer();
}

void TOpenGLDisplay::Set_Touch_Buffer_Size(
	const int							buffer_size)
{
	assert(!(buffer_size < 0));
	
	d_touch_buffer_size = buffer_size;
	
	this->Update_Path_Buffer();
}

void TOpenGLDisplay::GLInit(void)
{	
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glShadeModel(GL_SMOOTH);
	
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
	
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	d_triangle_shader_program = new QOpenGLShaderProgram(this);
	d_triangle_shader_program->addShaderFromSourceCode(QOpenGLShader::Vertex,OpenGLShaders::triangle_vertex_shader);
	d_triangle_shader_program->addShaderFromSourceCode(QOpenGLShader::Fragment,OpenGLShaders::triangle_fragment_shader);
	d_triangle_shader_program->link();
	
	d_triangle_uniform_color = d_triangle_shader_program->uniformLocation("color");
	d_triangle_uniform_specular = d_triangle_shader_program->uniformLocation("specular");
	d_triangle_uniform_offset = d_triangle_shader_program->uniformLocation("offset");
	d_triangle_attribute_vertex = d_triangle_shader_program->attributeLocation("vertex");
	d_triangle_attribute_normal = d_triangle_shader_program->attributeLocation("normal");
	
	d_basic_shader_program = new QOpenGLShaderProgram(this);
	d_basic_shader_program->addShaderFromSourceCode(QOpenGLShader::Vertex,OpenGLShaders::basic_vertex_shader);
	d_basic_shader_program->addShaderFromSourceCode(QOpenGLShader::Fragment,OpenGLShaders::basic_fragment_shader);
	d_basic_shader_program->link();
	
	d_basic_uniform_color = d_basic_shader_program->uniformLocation("color");
	d_basic_attribute_vertex = d_basic_shader_program->attributeLocation("vertex");
	
	glPointSize(5);

	glClearColor(1.0f,1.0f,1.0f,1.0f);
	
	this->Rotate_Model(TVector3(1,0,0),-1.31);	// 75 degrees
	this->Rotate_Model(TVector3(0,0,1),-0.26);	// 15 degrees
}

void TOpenGLDisplay::GLResize(
	const int 							width,
	const int 							height)
{
	Q_UNUSED(width);
	Q_UNUSED(height);
}

void TOpenGLDisplay::GLRender(void)
{
	std::vector<TVector3>::const_iterator pnt_iter;
	std::vector<TOpenGLDisplay::TTouchPoint>::iterator touch_iter;
	TVector3							pt1,pt2;
	TVector3							axis_display_positions[3];
	TVector3							pos;
	QString								text;
	QColor								label_color(224,224,224,160);
	int									text_x,text_y;
	unsigned int						ucntr;
	unsigned int						ucount;
	TVector3							pt;
	double								axis_lines[18];				// 3 lines, 2 points, 3 axis
	double								*vertex;
	
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	glEnable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glDisable(GL_CULL_FACE);
	
	// X Axis display line
		
	pt2 = d_axis_draw_pos + TVector3(1,0,0) * d_axis_draw_len;
	axis_display_positions[0] = pt2 + TVector3(1,0,0) * d_axis_draw_len/8;
	axis_lines[0] = d_axis_draw_pos.x;
	axis_lines[1] = d_axis_draw_pos.y;
	axis_lines[2] = d_axis_draw_pos.z;
	axis_lines[3] = pt2.x;
	axis_lines[4] = pt2.y;
	axis_lines[5] = pt2.z;
		
	// Y Axis display line
	
	pt2 = d_axis_draw_pos + TVector3(0,1,0) * d_axis_draw_len;
	axis_display_positions[1] = pt2 + TVector3(0,1,0) * d_axis_draw_len/8;
	axis_lines[6] = d_axis_draw_pos.x;
	axis_lines[7] = d_axis_draw_pos.y;
	axis_lines[8] = d_axis_draw_pos.z;
	axis_lines[9] = pt2.x;
	axis_lines[10] = pt2.y;
	axis_lines[11] = pt2.z;
	
	// Z Axis display line
	
	pt2 = d_axis_draw_pos + TVector3(0,0,1) * d_axis_draw_len;
	axis_display_positions[2] = pt2 + TVector3(0,0,1) * d_axis_draw_len/8;
	axis_lines[12] = d_axis_draw_pos.x;
	axis_lines[13] = d_axis_draw_pos.y;
	axis_lines[14] = d_axis_draw_pos.z;
	axis_lines[15] = pt2.x;
	axis_lines[16] = pt2.y;
	axis_lines[17] = pt2.z;

	if(this->Is_Processing_Scale_To_Fit())
	{
		this->Process_Vertex(d_axis_draw_pos);
		this->Process_Vertex(axis_lines[3],axis_lines[4],axis_lines[5]);
		this->Process_Vertex(axis_lines[9],axis_lines[10],axis_lines[11]);
		this->Process_Vertex(axis_lines[15],axis_lines[16],axis_lines[17]);
		
		if(!(d_touch_points_id < 0))
		{
			vertex = d_opengl_vertex_buffer.Buffer(d_touch_points_id);
			ucount = d_opengl_vertex_buffer.Buffer_Size(d_touch_points_id);
			
			if(vertex)
			{
				for(ucntr = 0;ucntr < ucount;ucntr += 3)
				{
					this->Process_Vertex(vertex[ucntr],vertex[ucntr+1],vertex[ucntr+2]);
				}
			}
		}
	}
	else
	{
		d_basic_shader_program->bind();
		
		if(!(d_touch_points_id < 0))
		{
			this->Render_Point_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_touch_points_id),TVector3(0.0,0.0,0.0),d_opengl_vertex_buffer.Buffer(d_touch_points_id));

			for(touch_iter = d_touch_buffer.begin();touch_iter != d_touch_buffer.end();++touch_iter)
			{
				this->Render_Line_Strip_Buffer((*touch_iter).path.size() * 3,TVector3(0.75,0.0,0.0),reinterpret_cast<double*>(&(*touch_iter).path[0]));
			}
		}
		
		this->Render_Line_Buffer(18,TVector3(0.25,0.25,0.25),axis_lines);
		
		
		d_basic_shader_program->release();
	}
 
	
	glEnable(GL_LIGHTING);
	glEnable(GL_CULL_FACE);
	
	if(this->Is_Processing_Scale_To_Fit())
	{
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_z_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_machine_z_vertex_id);
		
		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr] + d_machine_pos.x,vertex[ucntr+1] + d_machine_pos.y,vertex[ucntr+2] + d_machine_pos.z);
			}
		}
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_xz_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_machine_xz_vertex_id);
		
		pos = d_machine_pos;
		pos.z = d_max_xyz.z;

		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr] + pos.x,vertex[ucntr+1] + pos.y,vertex[ucntr+2] + pos.z);
			}
		}
		
		vertex = d_opengl_vertex_buffer.Buffer(d_machine_x_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_machine_x_vertex_id);
		
		pos = d_machine_pos;
		pos.x = 0.0;
		pos.z = d_max_xyz.z;
		
		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr] + pos.x,vertex[ucntr+1] + pos.y,vertex[ucntr+2] + pos.z);
			}
		}

		vertex = d_opengl_vertex_buffer.Buffer(d_machine_xw_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_machine_xw_vertex_id);
		
		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr] + pos.x,vertex[ucntr+1] + pos.y,vertex[ucntr+2] + pos.z);
			}
		}

		vertex = d_opengl_vertex_buffer.Buffer(d_machine_table_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_machine_table_vertex_id);
		
		pos.x = 0.0;
		pos.y = 0.0;
		pos.z = d_max_xyz.z;

		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr] + pos.x,vertex[ucntr+1] + pos.y,vertex[ucntr+2] + pos.z);
			}
		}
		
		vertex = d_opengl_vertex_buffer.Buffer(d_tool_stem_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_tool_stem_vertex_id);
		
		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr],vertex[ucntr+1],vertex[ucntr+2]);
			}
		}
	
		vertex = d_opengl_vertex_buffer.Buffer(d_probe_head_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_probe_head_vertex_id);
		
		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr],vertex[ucntr+1],vertex[ucntr+2]);
			}
		}
		
		vertex = d_opengl_vertex_buffer.Buffer(d_tool_ruby_vertex_id);
		ucount = d_opengl_vertex_buffer.Buffer_Size(d_tool_ruby_vertex_id);
		
		if(vertex)
		{
			for(ucntr = 0;ucntr < ucount;ucntr += 3)
			{
				this->Process_Vertex(vertex[ucntr],vertex[ucntr+1],vertex[ucntr+2]);
			}
		}
	}
	else
	{
		d_triangle_shader_program->bind();

		if(d_opengl_vertex_buffer.Has_Buffer(d_machine_z_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_machine_z_normal_id))
		{
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_machine_z_vertex_id),
								d_machine_pos,
								TVector3(0.25,0.25,0.25),
								0.75,
								d_opengl_vertex_buffer.Buffer(d_machine_z_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_machine_z_normal_id));

		}

		if(d_opengl_vertex_buffer.Has_Buffer(d_machine_xz_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_machine_xz_normal_id))
		{
			pos = d_machine_pos;
			pos.z = d_max_xyz.z;
			
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_machine_xz_vertex_id),
								pos,
								TVector3(0.80,0.80,0.80),
								0.1,
								d_opengl_vertex_buffer.Buffer(d_machine_xz_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_machine_xz_normal_id));

		}

		if(d_opengl_vertex_buffer.Has_Buffer(d_machine_x_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_machine_x_normal_id))
		{
			pos = d_machine_pos;
			pos.x = 0.0;
			pos.z = d_max_xyz.z;
			
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_machine_x_vertex_id),
								pos,
								TVector3(0.1,0.1,0.1),
								0.1,
								d_opengl_vertex_buffer.Buffer(d_machine_x_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_machine_x_normal_id));

		}

		if(d_opengl_vertex_buffer.Has_Buffer(d_machine_xw_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_machine_xw_normal_id))
		{
			pos = d_machine_pos;
			pos.x = 0.0;
			pos.z = d_max_xyz.z;
			
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_machine_xw_vertex_id),
								pos,
								TVector3(0.80,0.80,0.80),
								0.1,
								d_opengl_vertex_buffer.Buffer(d_machine_xw_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_machine_xw_normal_id));

		}

		if(d_opengl_vertex_buffer.Has_Buffer(d_machine_table_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_machine_table_normal_id))
		{
			pos.x = 0.0;
			pos.y = 0.0;
			pos.z = d_max_xyz.z;
			
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_machine_table_vertex_id),
								pos,
								TVector3(0.2,0.2,0.2),
								0.3,
								d_opengl_vertex_buffer.Buffer(d_machine_table_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_machine_table_normal_id));

		}

		if(d_opengl_vertex_buffer.Has_Buffer(d_tool_stem_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_tool_stem_normal_id))
		{
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_tool_stem_vertex_id),
								d_machine_pos,
								TVector3(0.7,0.7,0.7),
								0.25,
								d_opengl_vertex_buffer.Buffer(d_tool_stem_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_tool_stem_normal_id));
			
		}
		
		if(d_opengl_vertex_buffer.Has_Buffer(d_probe_head_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_probe_head_normal_id))
		{
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_probe_head_vertex_id),
								d_machine_pos,
								TVector3(0.2,0.2,0.2),
								0.25,
								d_opengl_vertex_buffer.Buffer(d_probe_head_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_probe_head_normal_id));
			
		}
		
		if(d_opengl_vertex_buffer.Has_Buffer(d_tool_ruby_vertex_id) && d_opengl_vertex_buffer.Has_Buffer(d_tool_ruby_normal_id))
		{
			this->Render_Triangle_Buffer(d_opengl_vertex_buffer.Buffer_Size(d_tool_ruby_vertex_id),
								d_machine_pos,
								TVector3(1.0,0.0,0.0),
								0.25,
								d_opengl_vertex_buffer.Buffer(d_tool_ruby_vertex_id),
								d_opengl_vertex_buffer.Buffer(d_tool_ruby_normal_id));
			
		}
		
		d_triangle_shader_program->release();
	}

	
	glDisable(GL_DEPTH_TEST);


	if(!this->Is_Processing_Scale_To_Fit())
	{
		// painting of text using QPainter
		// QPainter changes one or more attributes so all relevent ones
		// are pushed then popped on end of QPainter
		glPushAttrib(GL_ACCUM_BUFFER_BIT);
		glPushAttrib(GL_VIEWPORT_BIT);
		glPushAttrib(GL_TRANSFORM_BIT);
		glPushAttrib(GL_POLYGON_BIT);
		
		glPushAttrib(GL_PIXEL_MODE_BIT);
		glPushAttrib(GL_MULTISAMPLE_BIT);
		glPushAttrib(GL_LIGHTING_BIT);
		glPushAttrib(GL_ENABLE_BIT);
		
		glPushAttrib(GL_DEPTH_BUFFER_BIT);
		glPushAttrib(GL_CURRENT_BIT);
		glPushAttrib(GL_COLOR_BUFFER_BIT);

		// end of openGL, start of QPainter
		QPainter							painter(this);
		
		// axis labels
		this->Point_Raster_Position(&text_x,&text_y,axis_display_positions[0]);
		text = QStringLiteral("X");
		painter.drawText(text_x,text_y,text);
		
		this->Point_Raster_Position(&text_x,&text_y,axis_display_positions[1]);
		text = QStringLiteral("Y");
		painter.drawText(text_x,text_y,text);

		this->Point_Raster_Position(&text_x,&text_y,axis_display_positions[2]);
		text = QStringLiteral("Z");
		painter.drawText(text_x,text_y,text);

		painter.end();
		
		glPopAttrib();
		glPopAttrib();
		glPopAttrib();
		glPopAttrib();
		
		glPopAttrib();
		glPopAttrib();
		glPopAttrib();
		glPopAttrib();
		
		glPopAttrib();
		glPopAttrib();
		glPopAttrib();
	}
}

void TOpenGLDisplay::Update_Path_Buffer(void)
{
	std::vector<TOpenGLDisplay::TTouchPoint>::iterator touch_iter;
	double								*buffer;
	unsigned int						ucntr;
		
	if(!d_touch_buffer_size)
	{
		d_touch_buffer.clear();

		d_opengl_vertex_buffer.Remove_Buffer(d_touch_points_id);
		d_touch_points_id = -1;
		
		return;
	}
	
	while(d_touch_buffer.size() > static_cast<unsigned int>(d_touch_buffer_size))
	{
		d_touch_buffer.erase(d_touch_buffer.begin());
	}
	
	if(d_opengl_vertex_buffer.Buffer_Size(d_touch_points_id) != (d_touch_buffer.size() * 3))
	{
		d_opengl_vertex_buffer.Remove_Buffer(d_touch_points_id);
		d_touch_points_id = d_opengl_vertex_buffer.Create_Buffer(d_touch_buffer.size() * 3);
	}
	
	if(d_touch_points_id < 0)
	{
		return;
	}
	
	buffer = d_opengl_vertex_buffer.Buffer(d_touch_points_id);
	memset(buffer,0,d_touch_buffer.size() * 3 * sizeof(double));

	ucntr = 0;
	for(touch_iter = d_touch_buffer.begin();touch_iter != d_touch_buffer.end();++touch_iter)
	{
		buffer[ucntr] = (*touch_iter).pnt.x; ++ucntr;
		buffer[ucntr] = (*touch_iter).pnt.y; ++ucntr;
		buffer[ucntr] = (*touch_iter).pnt.z; ++ucntr;
	}
}

void TOpenGLDisplay::Render_Triangle_Buffer(
	const unsigned int 					size,
	const TVector3						&offset,
	const TVector3 						&color,
	const double 						&specular,
	double 								* const vertex,
	double 								* const nominal)
{
	d_triangle_shader_program->setUniformValue(d_triangle_uniform_color,color.x,color.y,color.z,1.0);
	d_triangle_shader_program->setUniformValue(d_triangle_uniform_offset,offset.x,offset.y,offset.z);
	d_triangle_shader_program->setUniformValue(d_triangle_uniform_specular,static_cast<float>(specular));
	
	glVertexAttribPointer(d_triangle_attribute_vertex, 3, GL_DOUBLE, GL_FALSE, 0, vertex);
	glVertexAttribPointer(d_triangle_attribute_normal, 3, GL_DOUBLE, GL_FALSE, 0, nominal);
	
	glEnableVertexAttribArray(d_triangle_attribute_vertex);
	glEnableVertexAttribArray(d_triangle_attribute_normal);
	
	glDrawArrays(GL_TRIANGLES, 0, size / 3);	// mode, first, index of 3 (x,y,z)
	
	glDisableVertexAttribArray(d_triangle_attribute_vertex);
	glDisableVertexAttribArray(d_triangle_attribute_normal);
}

void TOpenGLDisplay::Render_Line_Buffer(
	const unsigned int 					size,
	const TVector3 						&color,
	double 								* const vertex)
{
	d_basic_shader_program->setUniformValue(d_basic_uniform_color,color.x,color.y,color.z,1.0);
	
	glVertexAttribPointer(d_basic_attribute_vertex, 3, GL_DOUBLE, GL_FALSE, 0, vertex);
	
	glEnableVertexAttribArray(d_basic_attribute_vertex);
	
	glDrawArrays(GL_LINES, 0, size / 3);	// mode, first, index of 3 (x,y,z)
	
	glDisableVertexAttribArray(d_basic_attribute_vertex);
}

void TOpenGLDisplay::Render_Line_Strip_Buffer(
	const unsigned int 					size,
	const TVector3 						&color,
	double 								* const vertex)
{
	d_basic_shader_program->setUniformValue(d_basic_uniform_color,color.x,color.y,color.z,1.0);
	
	glVertexAttribPointer(d_basic_attribute_vertex, 3, GL_DOUBLE, GL_FALSE, 0, vertex);
	
	glEnableVertexAttribArray(d_basic_attribute_vertex);
	
	glDrawArrays(GL_LINE_STRIP, 0, size / 3);	// mode, first, index of 3 (x,y,z)
	
	glDisableVertexAttribArray(d_basic_attribute_vertex);
}

void TOpenGLDisplay::Render_Point_Buffer(
	const unsigned int 					size,
	const TVector3 						&color,
	double 								* const vertex)
{
	d_basic_shader_program->setUniformValue(d_basic_uniform_color,color.x,color.y,color.z,1.0);
	
	glVertexAttribPointer(d_basic_attribute_vertex, 3, GL_DOUBLE, GL_FALSE, 0, vertex);
	
	glEnableVertexAttribArray(d_basic_attribute_vertex);
	
	glDrawArrays(GL_POINTS, 0, size / 3);	// mode, first, index of 3 (x,y,z)
	
	glDisableVertexAttribArray(d_basic_attribute_vertex);
}


unsigned int TOpenGLDisplay::Convert_Geometry_To_Buffer(
	const std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> &data,
	double 								** const vertex,
	double 								** const normal) const
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>::const_iterator iter;
	int									cntr;
	unsigned int						size;

	if(*vertex)
	{
		delete[] (*vertex);
		(*vertex) = 0;
	}
	
	if(*normal)
	{
		delete[] (*normal);
		(*normal) = 0;
	}

	size = data.size() * 3;
	
	*vertex = new double[size];
	*normal = new double[size];
	
	cntr = 0;
	for(iter = data.begin();iter != data.end();++iter)
	{
		(*vertex)[cntr] = (*iter).vertex.x;
		(*vertex)[cntr+1] = (*iter).vertex.y;
		(*vertex)[cntr+2] = (*iter).vertex.z;
		
		(*normal)[cntr] = (*iter).normal.i;
		(*normal)[cntr+1] = (*iter).normal.j;
		(*normal)[cntr+2] = (*iter).normal.k;

		cntr += 3;
	}

	return size;
}


