/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QAction>
#include <QApplication>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QSpacerItem>
#include <QWhatsThis>
#include <QWidget>
#include <QComboBox>
#include <QProgressBar>
#include <QTimer>
#include <QScrollBar>
#include <QStringList>
#include <QMessageBox>
#include <QCloseEvent>

#include <assert.h>

#include "../../../core/optioneditor.h"

#include "protocoldialog.h"

TProtocolDialog::TProtocolDialog(
	const QWidget						*parent,
	Qt::WindowFlags 					flags)
: QDialog(const_cast<QWidget*>(parent),flags)
{
	QVBoxLayout 						*dialog_vspacer;
	QHBoxLayout 						*bottom_button_hlayout;
	QHBoxLayout 						*logtitle_hlayout;
	QSpacerItem 						*bottom_button_hspacer;
	QSpacerItem 						*logtitle_hspacer;

	this->resize(QSize(600,300));
	
	dialog_vspacer = new QVBoxLayout(this);
	
	d_option_editor = new TOptionEditorWidget(this);
	d_option_editor->headerItem()->setText(0, QStringLiteral("Option"));
	d_option_editor->headerItem()->setText(1, QStringLiteral("Value"));
	dialog_vspacer->addWidget(d_option_editor);

	logtitle_hlayout = new QHBoxLayout();
	
	logtitle_hspacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
	logtitle_hlayout->addItem(logtitle_hspacer);
	
	dialog_vspacer->addLayout(logtitle_hlayout);

	bottom_button_hlayout = new QHBoxLayout();
	
	bottom_button_hspacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
	bottom_button_hlayout->addItem(bottom_button_hspacer);
	
	d_cancel_button = new QPushButton(this);
	d_cancel_button->setAutoDefault(false);
	bottom_button_hlayout->addWidget(d_cancel_button);
	
	d_accept_button = new QPushButton(this);
	d_accept_button->setAutoDefault(false);
	bottom_button_hlayout->addWidget(d_accept_button);
	
	dialog_vspacer->addLayout(bottom_button_hlayout);
	
	QWidget::setTabOrder(d_option_editor, d_cancel_button);
	QWidget::setTabOrder(d_cancel_button, d_accept_button);
	
	d_cancel_button->setText(QStringLiteral("Cancel"));
	d_accept_button->setText(QStringLiteral("Accept"));
	
	// default values
	d_controller_type = TProtocolDialog::CONTROLLER_VIRTUALCMM;
	d_connection_type = TProtocolDialog::PROTOCOL_ETHERNET;
	d_serial_baudrate = QSerialPort::Baud9600;
	
	d_serial_portname = QStringLiteral("/dev/ttyUSB0");
#ifdef Q_OS_MAC
	d_serial_portname = QStringLiteral("/dev/ttys001");
#endif
#ifdef Q_OS_WIN
	d_serial_portname = QStringLiteral("COM1");
#endif
	
	d_ethernet_address = QStringLiteral("127.0.0.1");
	d_ethernet_port = 2001;

	this->setWindowTitle(QStringLiteral("Communication Options"));
	
	connect(d_option_editor,&TOptionEditorWidget::Option_Combo_Item_Changed,this,&TProtocolDialog::Option_Combo_Item_Changed);
	
	connect(d_cancel_button,&QPushButton::clicked,this,&TProtocolDialog::reject);
	connect(d_accept_button,&QPushButton::clicked,this,&TProtocolDialog::Accept);
}

TProtocolDialog::~TProtocolDialog()
{
	
}

TProtocolDialog::TControllerType TProtocolDialog::Controller_Type(void) const
{
	QString								text;
	TProtocolDialog::TControllerType	type(TProtocolDialog::CONTROLLER_VIRTUALCMM);
	
	text = d_option_editor->Option_Value(d_controller_type_index);
	
	if(text.contains(QStringLiteral("Leitz")))
	{
		type = TProtocolDialog::CONTROLLER_LEITZ;
	}
	else if(text.contains(QStringLiteral("DC")))
	{
		type = TProtocolDialog::CONTROLLER_DC;
	}
	else if(text.contains(QStringLiteral("I++")))
	{
		type = TProtocolDialog::CONTROLLER_IPP;
	}
	
	return type;
}

TProtocolDialog::TProtocolType TProtocolDialog::Connection_Type(void) const
{
	QString								text;
	TProtocolDialog::TProtocolType		type(TProtocolDialog::PROTOCOL_SERIAL);
	
	text = d_option_editor->Option_Value(d_protocol_type_index);
	
	if(text.contains(QStringLiteral("Ethernet")))
	{
		type = TProtocolDialog::PROTOCOL_ETHERNET;
	}
	
	return type;
}

QString TProtocolDialog::Comm_Port(void) const
{
	return d_option_editor->Option_Value(d_serial_comport_index);
}

QSerialPort::BaudRate TProtocolDialog::Baud_Rate(void)
{
	return this->Convert_Baudrate_Option(d_option_editor->Option_Value(d_serial_baud_index));
}

QString TProtocolDialog::Ethernet_Address(void) const
{
	return d_option_editor->Option_Value(d_ethernet_hostname_index);
}

int TProtocolDialog::Ethernet_Port(void) const
{
	return d_option_editor->Option_Value(d_ethernet_portnumber_index).toInt();
}

void TProtocolDialog::Set_Controller_Type(
	const TProtocolDialog::TControllerType type)
{
	d_controller_type = type;
}

void TProtocolDialog::Set_Connection_Type(
	const TProtocolDialog::TProtocolType type)
{
	d_connection_type = type;
}

void TProtocolDialog::Set_Comm_Port(
	const QString						&port_name)
{
	d_serial_portname = port_name;
}

void TProtocolDialog::Set_Baud_Rate(
	const QSerialPort::BaudRate			baud_rate)
{
	d_serial_baudrate = baud_rate;
}

void TProtocolDialog::Set_Ethernet_Address(
	const QString						&ethernet_address)
{
	d_ethernet_address = ethernet_address;
}

void TProtocolDialog::Set_Ethernet_Port(
	const int							port)
{
	d_ethernet_port = port;
}

void TProtocolDialog::Reset(void)
{
	this->Create_Options();
}

void TProtocolDialog::Option_Combo_Item_Changed(
	int									index,
	const QString						&text)
{
	if(index == d_protocol_type_index)
	{
		if(text.contains(QStringLiteral("Serial"),Qt::CaseInsensitive))
		{
			d_option_editor->Enable_Option(d_serial_comport_index,true);
			d_option_editor->Enable_Option(d_serial_baud_index,true);
			d_option_editor->Enable_Option(d_ethernet_hostname_index,false);
			d_option_editor->Enable_Option(d_ethernet_portnumber_index,false);
		}
		else
		{
			d_option_editor->Enable_Option(d_serial_comport_index,false);
			d_option_editor->Enable_Option(d_serial_baud_index,false);
			d_option_editor->Enable_Option(d_ethernet_hostname_index,true);
			d_option_editor->Enable_Option(d_ethernet_portnumber_index,true);
		}
	}
}

void TProtocolDialog::Accept(void)
{
	this->setFocus();

	d_controller_type = this->Controller_Type();
	d_connection_type = this->Connection_Type();
	
	d_serial_portname = d_option_editor->Option_Value(d_serial_comport_index);
	d_ethernet_address = d_option_editor->Option_Value(d_ethernet_hostname_index);
	d_ethernet_port = d_option_editor->Option_Value(d_ethernet_portnumber_index).toInt();

	this->accept();
}

void TProtocolDialog::Create_Options(void)
{
	QStringList							connection_methods;
	QStringList							controller_types;
	QStringList							serial_baud_rates;
	int									controller_index(0);
	int									protocol_index(0);
	int									baud_index(0);

// clear and invalidate all options
	d_option_editor->clear();
	
	d_controller_type_index = -1;
	d_protocol_type_index = -1;
	d_serial_comport_index = -1;
	d_serial_baud_index = -1;
	d_ethernet_hostname_index = -1;
	d_ethernet_portnumber_index = -1;
	
	controller_types.push_back(QStringLiteral("Virtual CMM (default)"));
	controller_types.push_back(QStringLiteral("Leitz"));
	controller_types.push_back(QStringLiteral("DC"));
	controller_types.push_back(QStringLiteral("I++ Server"));

	connection_methods.push_back(QStringLiteral("Serial"));
	connection_methods.push_back(QStringLiteral("Ethernet"));
	
	serial_baud_rates.push_back(QStringLiteral("1200 Baud"));
	serial_baud_rates.push_back(QStringLiteral("2400 Baud"));
	serial_baud_rates.push_back(QStringLiteral("4800 Baud"));
	serial_baud_rates.push_back(QStringLiteral("9600 Baud"));
	serial_baud_rates.push_back(QStringLiteral("19200 Baud"));
	serial_baud_rates.push_back(QStringLiteral("38400 Baud"));
	serial_baud_rates.push_back(QStringLiteral("57600 Baud"));
	serial_baud_rates.push_back(QStringLiteral("115200 Baud"));
	
	switch(d_controller_type)
	{
		case TProtocolDialog::CONTROLLER_VIRTUALCMM:
			controller_index = 0;
			break;

		case TProtocolDialog::CONTROLLER_LEITZ:
			controller_index = 1;
			break;
			
		case TProtocolDialog::CONTROLLER_DC:
			controller_index = 2;
			break;
			
		case TProtocolDialog::CONTROLLER_IPP:
			controller_index = 3;
			break;
	}
	
	switch(d_connection_type)
	{
		case TProtocolDialog::PROTOCOL_SERIAL:
			protocol_index = 0;
			break;
			
		case TProtocolDialog::PROTOCOL_ETHERNET:
			protocol_index = 1;
			break;
	}

	switch(d_serial_baudrate)
	{
		case QSerialPort::Baud1200:
			baud_index = 0;
			break;
		case QSerialPort::Baud2400:
			baud_index = 1;
			break;
		case QSerialPort::Baud4800:
			baud_index = 2;
			break;
		case QSerialPort::Baud9600:
			baud_index = 3;
			break;
		case QSerialPort::Baud19200:
			baud_index = 4;
			break;
		case QSerialPort::Baud38400:
			baud_index = 5;
			break;
		case QSerialPort::Baud57600:
			baud_index = 6;
			break;
		case QSerialPort::Baud115200:
			baud_index = 7;
			break;
			
		default:
			break;
	}
		
	d_controller_type_index = d_option_editor->Add_Combobox_Option("Controller Type",controller_types,controller_index);
	d_protocol_type_index = d_option_editor->Add_Combobox_Option("Communication Method",connection_methods,protocol_index);
	d_serial_comport_index = d_option_editor->Add_Line_Editor_Option("Serial Comm Port",d_serial_portname);
	d_serial_baud_index = d_option_editor->Add_Combobox_Option("Baud Rate",serial_baud_rates,baud_index);
	d_ethernet_hostname_index = d_option_editor->Add_Line_Editor_Option("Controller IP",d_ethernet_address);
	d_ethernet_portnumber_index = d_option_editor->Add_Line_Editor_Option("Port Number",QString("%1").arg(d_ethernet_port));
	
	d_option_editor->Enable_Option(d_serial_comport_index,protocol_index == 0);
	d_option_editor->Enable_Option(d_serial_baud_index,protocol_index == 0);
	d_option_editor->Enable_Option(d_ethernet_hostname_index,protocol_index != 0);
	d_option_editor->Enable_Option(d_ethernet_portnumber_index,protocol_index != 0);

	d_option_editor->resizeColumnToContents(0);
	d_option_editor->resizeColumnToContents(1);
}

QSerialPort::BaudRate TProtocolDialog::Convert_Baudrate_Option(
	const QString						&option_text)
{	
	if(option_text.contains("115200"))
	{
		return QSerialPort::Baud115200;
	}
	else if(option_text.contains("57600"))
	{
		return QSerialPort::Baud57600;
	}
	else if(option_text.contains("38400"))
	{
		return QSerialPort::Baud38400;
	}
	else if(option_text.contains("19200"))
	{
		return QSerialPort::Baud19200;
	}
	else if(option_text.contains("9600"))
	{
		return QSerialPort::Baud9600;
	}
	else if(option_text.contains("4800"))
	{
		return QSerialPort::Baud4800;
	}
	else if(option_text.contains("2400"))
	{
		return QSerialPort::Baud2400;
	}
	else if(option_text.contains("1200"))
	{
		return QSerialPort::Baud1200;
	}
	
	return QSerialPort::Baud9600;
}
